<?php

/**
 * StatusPage Settings Data.
 *
 * Handles the settings data and sanitization.
 *
 * @package StatusPageWidget
 */

// Exit if accessed directly.
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Class StatusPage_Settings
 *
 * Handles the plugin settings data.
 */
class StatusPage_Settings
{

    /**
     * Option name for storing defaults.
     *
     * @var string
     */
    const OPTION_NAME = 'statuspage_widget_defaults';

    /**
     * Default settings.
     *
     * @var array
     */
    private static $default_settings = array(
        'slug'           => '',
        'theme'          => 'auto',
        'compact'        => false,
        'show_incidents' => false,
        'rtm'            => 60,
        'upd'            => 7,
        'refresh'        => 0,
        'disable_link'   => false,
        'render_mode'    => 'js',
    );

    /**
     * Get the base default settings array.
     *
     * @return array Default settings.
     */
    public static function get_default_settings()
    {
        return self::$default_settings;
    }

    /**
     * Sanitize settings.
     *
     * @param array $input Input settings.
     * @return array Sanitized settings.
     */
    public static function sanitize_settings($input)
    {
        $sanitized = array();

        // Sanitize slug - only allow a-z, 0-9, and single hyphens between characters
        if (isset($input['slug'])) {
            $slug = strtolower(trim($input['slug']));
            // Validate slug pattern
            if (self::is_valid_slug($slug)) {
                $sanitized['slug'] = $slug;
            } else {
                $sanitized['slug'] = '';
                add_settings_error(
                    self::OPTION_NAME,
                    'invalid_slug',
                    __('Invalid slug format. Only lowercase letters (a-z), numbers (0-9), and single hyphens between characters are allowed.', 'status-page-widget'),
                    'error'
                );
            }
        } else {
            $sanitized['slug'] = '';
        }

        // Sanitize theme
        $sanitized['theme'] = isset($input['theme']) && in_array($input['theme'], array('auto', 'light', 'dark'), true)
            ? $input['theme']
            : 'auto';

        // Sanitize render_mode
        $sanitized['render_mode'] = isset($input['render_mode']) && in_array($input['render_mode'], array('js', 'php'), true)
            ? $input['render_mode']
            : 'js';

        // Sanitize booleans
        $sanitized['compact'] = ! empty($input['compact']);
        $sanitized['show_incidents'] = ! empty($input['show_incidents']);
        $sanitized['disable_link'] = ! empty($input['disable_link']);

        // Sanitize numbers
        $sanitized['rtm'] = isset($input['rtm']) ? min(max(absint($input['rtm']), 0), 240) : 60;
        $sanitized['upd'] = isset($input['upd']) ? min(max(absint($input['upd']), 0), 30) : 7;
        $sanitized['refresh'] = isset($input['refresh']) ? min(max(absint($input['refresh']), 0), 300) : 0;

        return $sanitized;
    }

    /**
     * Validate a slug.
     *
     * @param string $slug The slug to validate.
     * @return bool Whether the slug is valid.
     */
    public static function is_valid_slug($slug)
    {
        if (empty($slug)) {
            return true; // Empty is allowed
        }

        // Pattern: lowercase letters/numbers, single hyphens between them only
        return (bool) preg_match('/^[a-z0-9]+(-[a-z0-9]+)*$/', $slug);
    }

    /**
     * Get the default settings merged with saved options.
     *
     * @return array Default settings.
     */
    public static function get_defaults()
    {
        $saved = get_option(self::OPTION_NAME, array());
        return wp_parse_args($saved, self::$default_settings);
    }

    /**
     * Get a specific default value.
     *
     * @param string $key Setting key.
     * @return mixed Setting value.
     */
    public static function get_default($key)
    {
        $defaults = self::get_defaults();
        return isset($defaults[$key]) ? $defaults[$key] : null;
    }
}
